<?php
require_once 'config.php';

// Adatbázis kapcsolat
$db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($db->connect_error) {
    die("Kapcsolódási hiba: " . $db->connect_error);
}
$db->set_charset("utf8mb4");

// Terem azonosító
$terem_id = '106';

function getLessonData($db, $terem_id) {
    $current_day = date('N');
    
    $query = "
    WITH PrioritizedSubstitutions AS (
        SELECT 
            h.*,
            CASE 
                WHEN h.jelleg LIKE 'megtartva%' THEN 1
                WHEN h.jelleg LIKE 'elmarad%' THEN 2
                ELSE 3
            END as priority,
            ROW_NUMBER() OVER (
                PARTITION BY h.terem, h.ora_sorszam 
                ORDER BY 
                    CASE 
                        WHEN h.jelleg LIKE 'megtartva%' THEN 1
                        WHEN h.jelleg LIKE 'elmarad%' THEN 2
                        ELSE 3
                    END
            ) as rn
        FROM helyettesitesek h
    )
    SELECT o.*, 
           h.helyettesito_tanar, 
           h.jelleg,
           h.hianyzo_tanar,
           h.tantargy AS h_tantargy,
           h.osztaly AS h_osztaly
    FROM orarend o
    LEFT JOIN PrioritizedSubstitutions h ON o.terem_id = h.terem 
        AND o.ora_sorszam = CAST(SUBSTRING_INDEX(h.ora_sorszam, '.', 1) AS UNSIGNED)
        AND h.rn = 1
    WHERE o.terem_id = ? AND o.nap = ?
    ORDER BY o.ora_sorszam
    ";
    
    $stmt = $db->prepare($query);
    $stmt->bind_param("si", $terem_id, $current_day);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $orarend = [];
    while ($row = $result->fetch_assoc()) {
        if ($row['jelleg']) {
            $row['eredeti_tantargy'] = $row['tantargy'];
            $row['eredeti_tanar'] = $row['tanar'];
            $row['tantargy'] = $row['h_tantargy'] ?: $row['tantargy'];
            $row['tanar'] = $row['helyettesito_tanar'] ?: $row['tanar'];
            $row['osztaly'] = $row['h_osztaly'] ?: $row['osztaly'];
        }
        $orarend[] = $row;
    }
    
    return $orarend;
}

// AJAX kérés kezelése
if (isset($_GET['action'])) {
    if ($_GET['action'] == 'get_lesson_data') {
        $lesson_data = getLessonData($db, $terem_id);
        header('Content-Type: application/json');
        echo json_encode($lesson_data);
        exit;
    } elseif ($_GET['action'] == 'get_server_time') {
        $current_time = new DateTime('now', new DateTimeZone('Europe/Budapest'));
        header('Content-Type: application/json');
        echo json_encode(['server_time' => $current_time->format('Y-m-d H:i:s')]);
        exit;
    }
}

$db->close();
?>

<!DOCTYPE html>
<html lang="hu">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Órarend - <?php echo $terem_id; ?> terem</title>
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        html, body {
            height: 100%;
            width: 100%;
            overflow: hidden;
        }
        body {
            font-family: Arial, sans-serif;
            background: linear-gradient(rgba(255, 255, 255, 0.9), rgba(255, 255, 255, 0.9)),
                        url("data:image/svg+xml,%3Csvg width='20' height='20' viewBox='0 0 20 20' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='%23dcffdc' fill-opacity='0.4'%3E%3Cpath d='M10 0l10 10-10 10L0 10z'/%3E%3C/g%3E%3C/svg%3E");
            display: flex;
            flex-direction: column;
            align-items: center;
            padding: 20px;
        }

        /* Íves girland */
        .garland {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            height: 40px;
            background: transparent;
            overflow: hidden;
        }

        .lights {
            position: absolute;
            width: 100%;
            height: 100%;
            background-image: 
                radial-gradient(circle at 10px 30px, red 5px, transparent 6px),
                radial-gradient(circle at 30px 20px, green 5px, transparent 6px),
                radial-gradient(circle at 50px 30px, yellow 5px, transparent 6px),
                radial-gradient(circle at 70px 20px, red 5px, transparent 6px),
                radial-gradient(circle at 90px 30px, green 5px, transparent 6px);
            background-size: 100px 40px;
            background-repeat: repeat-x;
            animation: twinkle 1.5s ease-in-out infinite alternate;
        }

        @keyframes twinkle {
            0% { filter: brightness(0.5); }
            100% { filter: brightness(1.2); }
        }

        .header {
            width: 100%;
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            margin-top: 40px;
            background: linear-gradient(135deg, rgba(196, 30, 58, 0.1), rgba(34, 139, 34, 0.1));
            border-radius: 10px;
            padding: 10px;
        }

        .logo-container {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .logo {
            max-width: 100px;
        }

        .container {
            width: 100%;
            max-width: 1000px;
            display: flex;
            flex-direction: column;
            flex-grow: 1;
        }

        .lesson-container {
            display: flex;
            justify-content: space-between;
            margin-bottom: 20px;
        }

        .section {
            background-color: white;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            padding: 20px;
            width: 48%;
        }

        #currentLesson {
            border-left: 10px solid #c41e3a;
        }

        #nextLesson {
            border-left: 10px solid #228B22;
        }

        h1, h2 {
            text-align: center;
            margin-bottom: 15px;
            color: #c41e3a;
        }

        .info-box {
            background-color: #f8f9ff;
            border-radius: 5px;
            padding: 10px;
            margin: 10px 0;
            border: 1px solid #e0e0e0;
            transition: background-color 0.3s ease;
        }

        .info-box:hover {
            background-color: #fff5f5;
        }

        #currentTime {
            text-align: right;
            font-size: 1.2em;
            color: #34495e;
            background: linear-gradient(45deg, rgba(196, 30, 58, 0.1), rgba(34, 139, 34, 0.1));
            padding: 10px;
            border-radius: 5px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        @media (max-width: 768px) {
            .lesson-container {
                flex-direction: column;
            }
            .section {
                width: 100%;
                margin-bottom: 20px;
            }
        }
    </style>
</head>
<body>
    <div class="garland">
        <div class="lights"></div>
    </div>
    <div class="header">
        <div style="width: 150px;"></div>
        <div class="logo-container">
            <span style="font-size: 40px;">🎄</span>
            <img src="https://kartyaigeny.kanizsay.edu.hu/image/kanizsay.png" alt="Kanizsay logo" class="logo">
            <span style="font-size: 40px;">🎄</span>
        </div>
        <div id="currentTime"></div>
    </div>
    <div class="container">
        <h1>Órarend - <?php echo $terem_id; ?> terem</h1>
        
        <div class="lesson-container">
            <div id="currentLesson" class="section">
                <h2>Jelenlegi óra/állapot</h2>
                <div id="currentLessonContent">
                    <p class="info-box">Betöltés...</p>
                </div>
            </div>
            
            <div id="nextLesson" class="section">
                <h2>Következő óra</h2>
                <div id="nextLessonContent">
                    <p class="info-box">Betöltés...</p>
                </div>
            </div>
        </div>
    </div>

    <script>
        let clientServerTimeDiff = 0;
        let lessonData = null;

        const currentScript = location.pathname.split('/').pop();

        function syncTime() {
            fetch(`${currentScript}?action=get_server_time`)
                .then(response => response.json())
                .then(data => {
                    const serverTime = new Date(data.server_time);
                    const localTime = new Date();
                    clientServerTimeDiff = serverTime - localTime;
                })
                .catch(error => console.error('Hiba az idő szinkronizálásakor:', error));
        }

        function getCurrentTime() {
            return new Date(new Date().getTime() + clientServerTimeDiff);
        }

        function updateTime() {
            const now = getCurrentTime();
            document.getElementById('currentTime').innerHTML = now.toLocaleString('hu-HU');
        }

        function fetchLessonData() {
            fetch(`${currentScript}?action=get_lesson_data`)
                .then(response => response.json())
                .then(data => {
                    lessonData = data;
                    console.log("Lekért órarendi adatok:", lessonData);
                    updateLessonData();
                })
                .catch(error => console.error('Hiba az órarend lekérdezésekor:', error));
        }

        function updateLessonData() {
            if (!lessonData) return;

            const currentTime = getCurrentTime();
            const currentStatus = getCurrentStatus(lessonData, currentTime);
            const nextLesson = findNextLesson(lessonData, currentTime);

            updateCurrentStatus(currentStatus);
            updateNextLesson(nextLesson);
        }

        function getCurrentStatus(lessons, currentTime) {
            for (let i = 0; i < lessons.length; i++) {
                const lesson = lessons[i];
                const kezdes = new Date(currentTime.toDateString() + ' ' + lesson.kezdes);
                const vege = new Date(currentTime.toDateString() + ' ' + lesson.vege);

                if (currentTime >= kezdes && currentTime < vege) {
                    return lesson.tantargy ? { type: 'lesson', lesson } : { type: 'gap', lesson };
                }

                if (i < lessons.length - 1) {
                    const nextLesson = lessons[i + 1];
                    const nextKezdes = new Date(currentTime.toDateString() + ' ' + nextLesson.kezdes);
                    if (currentTime >= vege && currentTime < nextKezdes) {
                        return { type: 'break', current: lesson, next: nextLesson };
                    }
                }
            }

            return { type: 'noLesson' };
        }

        function findNextLesson(lessons, currentTime) {
            return lessons.find(lesson => {
                const kezdes = new Date(currentTime.toDateString() + ' ' + lesson.kezdes);
                return kezdes > currentTime && lesson.tantargy;
            });
        }

        function updateCurrentStatus(status) {
            const currentLessonContent = document.getElementById('currentLessonContent');
            let content = '';

            switch (status.type) {
                case 'lesson':
                    const isSubstitution = status.lesson.jelleg;
                    content = `
                        <p class="info-box">⏱️ Idő: ${status.lesson.kezdes} - ${status.lesson.vege}</p>
                        <p class="info-box">🔢 Óra sorszáma: ${status.lesson.ora_sorszam}</p>
                        <p class="info-box">👨‍🏫 Tanár: ${status.lesson.tanar}</p>
                        <p class="info-box">📖 Tárgy: ${status.lesson.tantargy}</p>
                        <p class="info-box">👥 Osztály: ${status.lesson.osztaly}</p>
                        ${isSubstitution ? `
                            <p class="info-box"><strong>Jelleg: ${status.lesson.jelleg}</strong></p>
                        ` : ''}
                        ${status.lesson.hianyzo_tanar ? `<p class="info-box">Hiányzó tanár: ${status.lesson.hianyzo_tanar}</p>` : ''}
                    `;
                    break;
                case 'gap':
                    content = '<p class="info-box">Lyukas óra</p>';
                    break;
                case 'break':
                    content = `<p class="info-box">Jelenleg szünet van a teremben!</p>
                               <p class="info-box">Következő óra kezdete: ${status.next.kezdes}</p>`;
                    break;
                case 'noLesson':
                    content = '<p class="info-box">Ma már nincs több óra a teremben.</p>';
                    break;
            }

            currentLessonContent.innerHTML = content;
        }

        function updateNextLesson(lesson) {
            const nextLessonContent = document.getElementById('nextLessonContent');
            if (lesson) {
                const isSubstitution = lesson.jelleg;
                nextLessonContent.innerHTML = `
                    <p class="info-box">⏱️ Idő: ${lesson.kezdes} - ${lesson.vege}</p>
                    <p class="info-box">🔢 Óra sorszáma: ${lesson.ora_sorszam}</p>
                    <p class="info-box">👨‍🏫 Tanár: ${lesson.tanar}</p>
                    <p class="info-box">📖 Tárgy: ${lesson.tantargy}</p>
                    <p class="info-box">👥 Osztály: ${lesson.osztaly}</p>
                    ${isSubstitution ? `
                        <p class="info-box"><strong>Jelleg: ${lesson.jelleg}</strong></p>
                    ` : ''}
                    ${lesson.hianyzo_tanar ? `<p class="info-box">Hiányzó tanár: ${lesson.hianyzo_tanar}</p>` : ''}
                `;
            } else {
                nextLessonContent.innerHTML = `
                    <p class="info-box">Mára nincs több óra betervezve.</p>
                `;
            }
        }

        function handleDayNightTransition() {
            const now = getCurrentTime();
            const hour = now.getHours();
            const day = now.getDay(); // 0 = vasárnap, 6 = szombat
            
            // Elsötétítés ha:
            // - hétvége van (szombat vagy vasárnap) VAGY
            // - hétköznap 16:00 után vagy 7:00 előtt
            if (day === 0 || day === 6 || hour >= 16 || hour < 7) {
                document.body.style.transition = 'opacity 1s';
                document.body.style.opacity = '0';
            } else {
                document.body.style.opacity = '1';
            }
        }

        // Indítás
        syncTime();
        fetchLessonData();

        // Időzítők beállítása
        setInterval(updateTime, 1000);
        setInterval(updateLessonData, 5000);  // 5 másodpercenként frissít
        setInterval(syncTime, 900000);
        setInterval(fetchLessonData, 3600000);  // Óránként frissíti az órarendet
        setInterval(handleDayNightTransition, 60000); // Percenként ellenőrzi az időt

        // Kezdeti frissítések
        updateTime();
        handleDayNightTransition();
    </script>
</body>
</html>