<?php
require_once 'config.php';

// Adatbázis kapcsolat
$db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($db->connect_error) {
    die("Kapcsolódási hiba: " . $db->connect_error);
}
$db->set_charset("utf8mb4");

// Terem azonosító
$terem_id = 'I/10';

function getLessonData($db, $terem_id) {
    $current_day = date('N');
    $current_week = date('W');
    $is_even_week = ($current_week % 2 == 0); // Páros hét

    $query = "
    WITH PrioritizedSubstitutions AS (
        SELECT 
            h.*,
            CASE 
                WHEN h.jelleg LIKE 'megtartva%' THEN 1
                WHEN h.jelleg LIKE 'elmarad%' THEN 2
                ELSE 3
            END as priority,
            ROW_NUMBER() OVER (
                PARTITION BY h.terem, h.ora_sorszam 
                ORDER BY 
                    CASE 
                        WHEN h.jelleg LIKE 'megtartva%' THEN 1
                        WHEN h.jelleg LIKE 'elmarad%' THEN 2
                        ELSE 3
                    END
            ) as rn
        FROM helyettesitesek h
    )
    SELECT o.*, 
           h.helyettesito_tanar, 
           h.jelleg,
           h.hianyzo_tanar,
           h.tantargy AS h_tantargy,
           h.osztaly AS h_osztaly
    FROM orarend o
    LEFT JOIN PrioritizedSubstitutions h ON o.terem_id = h.terem 
        AND o.ora_sorszam = CAST(SUBSTRING_INDEX(h.ora_sorszam, '.', 1) AS UNSIGNED)
        AND h.rn = 1
    WHERE o.terem_id = ? AND o.nap = ?
    AND (o.het IS NULL OR o.het = ?)
    ORDER BY o.ora_sorszam
    ";
    
    $stmt = $db->prepare($query);
    $stmt->bind_param("sii", $terem_id, $current_day, $is_even_week);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $orarend = [];
    while ($row = $result->fetch_assoc()) {
        if ($row['jelleg']) {
            $row['eredeti_tantargy'] = $row['tantargy'];
            $row['eredeti_tanar'] = $row['tanar'];
            $row['tantargy'] = $row['h_tantargy'] ?: $row['tantargy'];
            $row['tanar'] = $row['helyettesito_tanar'] ?: $row['tanar'];
            $row['osztaly'] = $row['h_osztaly'] ?: $row['osztaly'];
        }
        $orarend[] = $row;
    }
    
    return $orarend;
}

// AJAX kérés kezelése
if (isset($_GET['action'])) {
    if ($_GET['action'] == 'get_lesson_data') {
        $lesson_data = getLessonData($db, $terem_id);
        header('Content-Type: application/json');
        echo json_encode($lesson_data);
        exit;
    } elseif ($_GET['action'] == 'get_server_time') {
        $current_time = new DateTime('now', new DateTimeZone('Europe/Budapest'));
        header('Content-Type: application/json');
        echo json_encode(['server_time' => $current_time->format('Y-m-d H:i:s')]);
        exit;
    }
}

$db->close();
?>
<!DOCTYPE html>
<html lang="hu">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Órarend - <?php echo $terem_id; ?> terem</title>
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        html, body {
            height: 100%;
            width: 100%;
            overflow: hidden;
        }

        body {
            font-family: serif;
            background: #F8F6F0;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 20px;
            position: relative;
            color: #7D5A50;
        }

        /* Halvány rácsos háttér */
        body::before {
            content: "";
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-image: repeating-linear-gradient(0deg, transparent, transparent 29px, #D9D2C5 29px, #D9D2C5 30px),
                              repeating-linear-gradient(90deg, transparent, transparent 29px, #D9D2C5 29px, #D9D2C5 30px);
            opacity: 0.2;
            z-index: -1;
        }

        /* Fő tároló konténer */
        .main-container {
            width: 95%;
            max-width: 1200px;
            height: auto;
            position: relative;
            display: flex;
            flex-direction: column;
            border: double 5px #7D5A50;
            padding: 15px;
            background-color: #F8F6F0;
            aspect-ratio: 1.6/1; /* Szélesebb, mint magasabb arány */
        }

        /* Belső keret */
        .inner-container {
            position: relative;
            width: 100%;
            height: 100%;
            border: 1px solid #7D5A50;
            padding: 15px;
            display: flex;
            flex-direction: column;
        }

        /* Fejléc terület */
        .header {
            width: 100%;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 5px 10px;
            border-bottom: 1px solid #7D5A50;
            margin-bottom: 15px;
        }

        .quote-container {
            font-weight: bold;
            font-size: 18px;
            color: #7D5A50;
            padding: 5px 10px;
            border: 1px solid #7D5A50;
            background-color: #F3EEE7;
            font-style: italic;
        }

        .logo-container {
            text-align: center;
            padding: 5px;
        }

        .logo {
            max-height: 70px;
            transition: transform 0.3s;
        }

        .time-container {
            text-align: center;
            padding: 5px 10px;
            border: 1px solid #7D5A50;
            background-color: #F3EEE7;
        }

        /* Cím */
        h1 {
            text-align: center;
            margin: 5px 0 15px;
            color: #7D5A50;
            font-size: 2.2em;
            font-weight: bold;
        }

        /* Elválasztó vonal díszítéssel */
        .divider {
            position: relative;
            height: 20px;
            margin: 5px 0;
            text-align: center;
        }

        .divider::before {
            content: "";
            position: absolute;
            left: 0;
            right: 0;
            top: 50%;
            height: 1px;
            background-color: #7D5A50;
        }

        .divider::after {
            content: "❀";
            background-color: #F8F6F0;
            color: #7D5A50;
            font-size: 16px;
            padding: 0 10px;
            position: relative;
            display: inline-block;
        }

        /* Órarendi konténer - kompaktabb, kevesebb függőleges hely */
        .lesson-container {
            display: flex;
            justify-content: space-between;
            gap: 30px;
            flex-grow: 0;
            margin-bottom: 10px;
        }

        /* Szekció címek */
        h2 {
            text-align: center;
            margin-bottom: 10px;
            color: #7D5A50;
            font-size: 1.3em;
            padding-bottom: 5px;
            border-bottom: 1px dashed #7D5A50;
            font-weight: bold;
        }

        /* Az órarendi információkat tartalmazó szekciók */
        .section {
            background-color: #F3EEE7;
            border: 1px solid #7D5A50;
            padding: 10px 15px;
            width: 48%;
            display: flex;
            flex-direction: column;
        }

        /* Információs dobozok (órarendi elemek) - nagyobb szöveg */
        .info-box {
            background-color: #F8F6F0;
            padding: 10px 15px;
            margin: 6px 0;
            border: 1px dashed #B85042;
            color: #7D5A50;
            font-size: 18px; /* Nagyobb szövegméret */
        }

        #currentTime, #currentDate {
            font-size: 18px; /* Nagyobb szövegméret */
            margin: 2px 0;
        }

        /* Évszám az oldal alján */
        .graduation-year {
            text-align: center;
            font-weight: bold;
            font-size: 20px; /* Nagyobb szövegméret */
            color: #7D5A50;
            letter-spacing: 10px;
            margin: 5px 0;
        }

        @media (max-width: 1280px) {
            .main-container {
                width: 98%;
                aspect-ratio: 1.5/1;
            }
            
            .info-box {
                padding: 8px 10px;
                font-size: 16px;
                margin: 4px 0;
            }
            
            h1 {
                margin: 5px 0 10px;
                font-size: 1.8em;
            }
            
            h2 {
                margin-bottom: 8px;
                font-size: 1.2em;
            }
            
            .quote-container {
                font-size: 16px;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <div class="inner-container">
            <div class="header">
                <div class="quote-container">Ballag már az vén diák...</div>
                
                <div class="logo-container">
                    <img src="https://kartyaigeny.kanizsay.edu.hu/image/kanizsay.png" alt="Kanizsay logo" class="logo">
                </div>
                
                <div class="time-container">
                    <div id="currentTime">10:07:21</div>
                    <div id="currentDate">2025. március 24., hétfő</div>
                </div>
            </div>
            
            <h1>Órarend - <?php echo $terem_id; ?> terem</h1>
            
            <div class="divider"></div>
            
            <div class="lesson-container">
                <div id="currentLesson" class="section">
                    <h2>Jelenlegi óra/állapot</h2>
                    <div id="currentLessonContent">
                        <p class="info-box">⏱️ Idő: 10:05 - 10:50</p>
                        <p class="info-box">🔢 Óra sorszáma: 3</p>
                        <p class="info-box">👨‍🏫 Tanár: Simon Mirjam</p>
                        <p class="info-box">📖 Tárgy: Matematika A2</p>
                        <p class="info-box">👥 Osztály: 9.A</p>
                    </div>
                </div>
                
                <div id="nextLesson" class="section">
                    <h2>Következő óra</h2>
                    <div id="nextLessonContent">
                        <p class="info-box">⏱️ Idő: 11:00 - 11:45</p>
                        <p class="info-box">🔢 Óra sorszáma: 4</p>
                        <p class="info-box">👨‍🏫 Tanár: Varga Zoltán Tamás</p>
                        <p class="info-box">📖 Tárgy: Magyar nyelv és irodalom alap</p>
                        <p class="info-box">👥 Osztály: 9.A</p>
                    </div>
                </div>
            </div>
            
            <div class="divider"></div>
            
            <div class="graduation-year">2 0 2 0 / 2 0 2 1 - 2 0 2 5</div>
        </div>
    </div>

    <script>
        let clientServerTimeDiff = 0;
        let lessonData = null;

        const currentScript = location.pathname.split('/').pop();

        function syncTime() {
            fetch(`${currentScript}?action=get_server_time`)
                .then(response => response.json())
                .then(data => {
                    const serverTime = new Date(data.server_time);
                    const localTime = new Date();
                    clientServerTimeDiff = serverTime - localTime;
                })
                .catch(error => console.error('Hiba az idő szinkronizálásakor:', error));
        }

        function getCurrentTime() {
            return new Date(new Date().getTime() + clientServerTimeDiff);
        }

        function updateTime() {
            const now = getCurrentTime();
            const timeStr = now.toLocaleTimeString('hu-HU');
            const dateStr = now.toLocaleDateString('hu-HU', { 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                weekday: 'long'
            });
            document.getElementById('currentTime').innerHTML = timeStr;
            document.getElementById('currentDate').innerHTML = dateStr;
        }

        function fetchLessonData() {
            fetch(`${currentScript}?action=get_lesson_data`)
                .then(response => response.json())
                .then(data => {
                    lessonData = data;
                    updateLessonData();
                })
                .catch(error => console.error('Hiba az órarend lekérdezésekor:', error));
        }

        function updateLessonData() {
            if (!lessonData) return;

            const currentTime = getCurrentTime();
            const currentStatus = getCurrentStatus(lessonData, currentTime);
            const nextLesson = findNextLesson(lessonData, currentTime);

            updateCurrentStatus(currentStatus);
            updateNextLesson(nextLesson);
        }

        function getCurrentStatus(lessons, currentTime) {
            for (let i = 0; i < lessons.length; i++) {
                const lesson = lessons[i];
                const kezdes = new Date(currentTime.toDateString() + ' ' + lesson.kezdes);
                const vege = new Date(currentTime.toDateString() + ' ' + lesson.vege);

                if (currentTime >= kezdes && currentTime < vege) {
                    return lesson.tantargy ? { type: 'lesson', lesson } : { type: 'gap', lesson };
                }

                if (i < lessons.length - 1) {
                    const nextLesson = lessons[i + 1];
                    const nextKezdes = new Date(currentTime.toDateString() + ' ' + nextLesson.kezdes);
                    if (currentTime >= vege && currentTime < nextKezdes) {
                        return { type: 'break', current: lesson, next: nextLesson };
                    }
                }
            }

            return { type: 'noLesson' };
        }

        function findNextLesson(lessons, currentTime) {
            return lessons.find(lesson => {
                const kezdes = new Date(currentTime.toDateString() + ' ' + lesson.kezdes);
                return kezdes > currentTime && lesson.tantargy;
            });
        }

        function updateCurrentStatus(status) {
            const currentLessonContent = document.getElementById('currentLessonContent');
            let content = '';

            switch (status.type) {
                case 'lesson':
                    const isSubstitution = status.lesson.jelleg;
                    content = `
                        <p class="info-box">⏱️ Idő: ${status.lesson.kezdes} - ${status.lesson.vege}</p>
                        <p class="info-box">🔢 Óra sorszáma: ${status.lesson.ora_sorszam}</p>
                        <p class="info-box">👨‍🏫 Tanár: ${status.lesson.tanar}</p>
                        <p class="info-box">📖 Tárgy: ${status.lesson.tantargy}</p>
                        <p class="info-box">👥 Osztály: ${status.lesson.osztaly}</p>
                        ${isSubstitution ? `
                            <p class="info-box"><strong>Jelleg: ${status.lesson.jelleg}</strong></p>
                        ` : ''}
                        ${status.lesson.hianyzo_tanar ? `<p class="info-box">Hiányzó tanár: ${status.lesson.hianyzo_tanar}</p>` : ''}
                    `;
                    break;
                case 'gap':
                    content = '<p class="info-box">Lyukas óra</p>';
                    break;
                case 'break':
                    content = `<p class="info-box">Jelenleg szünet van a teremben!</p>
                               <p class="info-box">Következő óra kezdete: ${status.next.kezdes}</p>`;
                    break;
                case 'noLesson':
                    content = '<p class="info-box">Ma már nincs több óra a teremben.</p>';
                    break;
            }

            currentLessonContent.innerHTML = content;
        }

        function updateNextLesson(lesson) {
            const nextLessonContent = document.getElementById('nextLessonContent');
            if (lesson) {
                const isSubstitution = lesson.jelleg;
                nextLessonContent.innerHTML = `
                    <p class="info-box">⏱️ Idő: ${lesson.kezdes} - ${lesson.vege}</p>
                    <p class="info-box">🔢 Óra sorszáma: ${lesson.ora_sorszam}</p>
                    <p class="info-box">👨‍🏫 Tanár: ${lesson.tanar}</p>
                    <p class="info-box">📖 Tárgy: ${lesson.tantargy}</p>
                    <p class="info-box">👥 Osztály: ${lesson.osztaly}</p>
                    ${isSubstitution ? `
                        <p class="info-box"><strong>Jelleg: ${lesson.jelleg}</strong></p>
                    ` : ''}
                    ${lesson.hianyzo_tanar ? `<p class="info-box">Hiányzó tanár: ${lesson.hianyzo_tanar}</p>` : ''}
                `;
            } else {
                nextLessonContent.innerHTML = `
                    <p class="info-box">Mára nincs több óra betervezve.</p>
                `;
            }
        }

        // Inicializálás
        syncTime();
        fetchLessonData();

        setInterval(updateTime, 1000);
        setInterval(updateLessonData, 5000);
        setInterval(syncTime, 900000);
        
        // Óránkénti frissítés eltávolítva a kérésnek megfelelően
        // setInterval(fetchLessonData, 3600000);

        updateTime();
    </script>
</body>
</html>